########### Clean Raw data ###########
# This function takes raw traces exported from NEX and will:
# - clean data by removing empty traces and units that do not pass the minimal threshold (defined in: "min.spikes")
# - Seperate the sessions, add its ID, and add trial periods (i.e. precue period, ITI period)
# - Add brain area ID (i.e. STN and/or SN, taken from file name)
# - Add Treatment ID (i.e. Baseline, Vehicle)
# - Multiply the value in each bin. Config: "multiplier"
# - Perform smoothing if needed (run moving average Gaussian smoothing on the entire neuron trace). Config: "smooth.spikes" and "smooth.window"
# - Normalize data if needed by either standard z-score, or Poisson approximation z-score. Config: "normalize.spikes.per.session" <- T/F; "normalize.spikes.method" <- z.score / poisson.zscore
# - Merge files from all animals into one RDS file (long format)

clean_raw_traces <- function(file.pattern = "traces.csv",
                             config.file = './R/config.R'){
  tryCatch({
    
  # Load libraries
  suppressWarnings(library(tidyverse))
  suppressWarnings(library(futile.logger))
  suppressWarnings(library(smoother))

  # Load config file
  source(config.file)
  source('./R/1_clean_raw_functions.R')

  # Set up logger
  flog.logger(name = log.name, threshold = threshold, appender = appender.tee(paste0(output.folder, log.name, ".log")))
  flog.threshold(threshold, name = log.name)
  
  flog.info(paste0("Initializing: ", format(Sys.Date(), "%d %B, %Y")), name = log.name)
  
  # Make list of files
  file.list <- dir(path = raw.data.folder, pattern = file.pattern)
  
  Merged.data <- vector(mode = "list", length = length(file.list))
  set_names(Merged.data, file.list)
  
  # Iterate through files
  for (file in file.list) {
    # init
    Seperated.sessions <- vector(mode = "list", length = 0)
    Session.timing <- data.frame()
    Data <- data.frame()
    
    # start data
    
    Data <- load_raw(file.name = file, raw.data.folder = raw.data.folder)
    
    
    Session.timing <- get_session(Data)
    
    # spererate sessions
    Seperated.sessions <- map(Session.timing$Session, 
                              seperate_session, # MAIN function
                              Session.timing = Session.timing, 
                              Data = Data, 
                              min.spikes = min.spikes,
                              z.score.session = normalize.spikes.per.session,
                              smooth.spikes = smooth.spikes, smooth.window = smooth.window,
                              spike.multiplier = multiplier,
                              normalize.spikes.method = normalize.spikes.method)
    
    Seperated.sessions <- map(Seperated.sessions, add_period)
    
    Cleaned.data <- map_df(Seperated.sessions, add_area_id)
    
    Merged.data[[file]] <- Cleaned.data
    flog.info(paste0("Finished file: ", file), name = log.name)
    
  }
  
  flog.info("Mergins and saving final data...", name = log.name)
  Final.data <- bind_rows(Merged.data)
  
  Final.data <- Final.data %>%
    mutate_at(vars(starts_with("Motion")), funs(as.numeric)) %>%
    mutate_at(vars(contains("_position")), funs(as.numeric)) %>%
    mutate_at(vars(starts_with("Freezing_")), funs(as.numeric))
  
  flog.info("Removing non filtered units.", name = log.name)
  Final.data <- Final.data %>% filter(!str_detect(Unit, pattern = "SPK..i"))
  
  flog.info("Removing units from list", name = log.name)
  Final.data <- Final.data %>% 
    unite(UnitID, File.name, SessionID, Unit, remove = FALSE) %>%
    filter(!(UnitID %in% remove.units)) %>% 
    select(-UnitID)
  
  write_rds(Final.data, path = paste0(output.folder, cleaned.raw.data), "gz", compression = 9L)

  flog.info("It is done!", name = log.name)
  
  return(NULL)

  }, error = function(e) {
    flog.error("ERROR!", name = log.name)
    flog.error(e, name = log.name)
  })
}
